# This file is example of deploy.py
#
# Path to this file is configured in web.config:
#      <application name="django.project.x86" >
#        <environmentVariables>
#          <add name="DEPLOY_FILE" value="deploy.py" />
#          <add name="DEPLOY_LOG"  value="log\deploy.log" />
#        </environmentVariables>
#      </application>
#
# The file is executed once on the first request after every restart of IIS application.
# The file output is redirected to log file described in DEPLOG_LOG environment variable.
#

import sys
import os
import os.path
import shutil
import msvcrt
import time
import re
import urllib2
from contextlib import closing
import tarfile
import errno


class Deployer:

    class Unbuffered:
        def __init__(self, stream):
            self.stream = stream

        def write(self, data):
            self.stream.write(data)
            self.stream.flush()

        def __getattr__(self, attr):
            return getattr(self.stream, attr)

    DOWNLOAD_CHUNK = 1024*1024
    DOWNLOAD_PROGRESS_STEP_DOT = 1*1024*1024
    DOWNLOAD_PROGRESS_NUM_EVERY_DOT = 10

    DISABLE_ZOO_WEB_CONFIG = """<?xml version="1.0" encoding="UTF-8"?>
<configuration>
    <system.webServer>
        <handlers>
            <remove name="{0}#x86" />
            <remove name="{0}#x64" />
        </handlers>
    </system.webServer>
</configuration>
"""

    def __init__(self):
        try:
            self._disable_std_streams_buffering()
        except Exception:
            pass

        self.app_dir = os.environ.get('APPL_PHYSICAL_PATH') or os.path.dirname(__file__)
        self.virtualenv_exe = os.path.join(os.path.dirname(sys.executable), 'scripts\\virtualenv.exe')
        self.virtualenv_name = 'venv'
        self.virtualenv_dir = os.path.join(self.app_dir, self.virtualenv_name)
        self.local_python = os.path.join(self.virtualenv_dir, 'scripts', 'python.exe')
        self.pip_exe = os.path.join(self.virtualenv_dir, 'scripts', 'pip.exe')

        self.last_progress_print = 0
        self.dirs = []

        os.chdir(self.app_dir)

        # update APPDATA env for pip
        os.environ['APPDATA'] = self.virtualenv_dir

        # add to python path:
        p1 = os.path.abspath(os.path.join(self.virtualenv_dir, 'Lib'))
        sys.path.insert(0, p1)
        os.environ['PYTHONPATH'] = p1 + ';' + os.environ.get('PYTHONPATH', '')
        p2 = os.path.abspath(os.path.join(self.virtualenv_dir, 'Lib', 'site-packages'))
        sys.path.insert(0, p2)
        os.environ['PYTHONPATH'] = p2 + ';' + os.environ.get('PYTHONPATH', '')

        # add scripts folder to PATH
        self.scripts_folder = os.path.abspath(os.path.join(self.virtualenv_dir, 'Scripts'))
        if os.environ.get('PATH'):
            os.environ['PATH'] = self.scripts_folder + ';' + os.environ['PATH']
        
        os.environ['VIRTUAL_ENV'] = self.virtualenv_dir

    def make_virtualenv(self, name=None):
        if not name:
            name = self.virtualenv_name
        if not os.path.exists(name):
            self.system(self.virtualenv_exe + ' ' + name)

    def _disable_std_streams_buffering(self):
        sys.stdout = Deployer.Unbuffered(sys.stdout)
        sys.stderr = Deployer.Unbuffered(sys.stderr)
        msvcrt.setmode(sys.stdout.fileno(), os.O_BINARY)
        msvcrt.setmode(sys.stderr.fileno(), os.O_BINARY)

    def mkdir_p(self, path):
        try:
            os.makedirs(path)
        except OSError as exc:
            if exc.errno == errno.EEXIST:
                pass
            else:
                raise

    def pushd(self, dirname):
        self.dirs.append(os.getcwd())
        os.chdir(dirname)

    def popd(self):
        if self.dirs:
            os.chdir(self.dirs.pop())

    def rmfile(self, file):
        if os.path.exists(file):
            os.remove(file)

    def system(self, command):
        print('\nRunning command: '+command)
        status = os.system(command)
        if status != 0:
            sys.stderr.write('Command "{0}" returns exit code {1}'.format(command, status))
            sys.exit(status)

    def run_with_local_python(self, script):
        self.system(self.local_python + ' ' + script)

    def path_in_app(self, *args):
        return os.path.join(self.app_dir, *args)

    def install_requirements(self, requirements_path='requirements.txt'):
        print("Installing requirements")
        self.system('{0} install --requirement={1}'.format(
            self.pip_exe,
            requirements_path
        ))

    def save_file(self, path, content):
        p = self.path_in_app(path)
        print("Updating file {0}".format(p))
        with open(p, 'wb') as f:
            f.write(content)

    def get_filename_from_url(self, url):
        if url.find('/') > 0:
            filename = url.split('/')[-1]
            if len(filename) > 4:
                return filename
        return re.sub(r'[-/:_\?&]+', '_', url)

    def download(self, url, filename=None):
        target = self.path_in_app(filename or self.get_filename_from_url(url))
        if os.path.exists(target):
            print('File {0} already exists. Download is skipped.'.format(target))
            return target
        print("Downloading "+url)
        chunk=None
        total = 0
        self.last_progress_print = 0
        with closing(urllib2.urlopen(url)) as furl:
            with open(target, 'wb') as fout:
                while True:
                    chunk = furl.read(self.DOWNLOAD_CHUNK)
                    if not chunk:
                        break
                    fout.write(chunk)
                    total += len(chunk)
                    self._print_download_progress(total)
        print('\nDownloaded {0:.1f} Mb'.format(total/1024.0/1024.0))
        print('Saved to '+target)
        return target

    def _print_download_progress(self, bytes_downloaded):
        delta_dots = (bytes_downloaded - self.last_progress_print*self.DOWNLOAD_PROGRESS_STEP_DOT)/self.DOWNLOAD_PROGRESS_STEP_DOT
        if delta_dots > 0:
            while delta_dots:
                delta_dots -= 1
                sys.stdout.write('.')
                self.last_progress_print += 1
                if self.last_progress_print % self.DOWNLOAD_PROGRESS_NUM_EVERY_DOT == 0:
                    sys.stdout.write(' {0} Mb\n'.format(bytes_downloaded/1024/1024))
                sys.stdout.flush()

    def create_django_user(self, settings, username, password, is_admin=True):
        try:
            from django.core.management import setup_environ
            setup_environ(settings)
            from django.contrib.auth.models import User
            # check superuser exists
            admins = User.objects.filter(username=username)
            if admins.count() >= 1:
                print('Django user already exists: {0}'.format(admins))
            else:
                admin = User()
                admin.username = username
                admin.is_staff = is_admin
                admin.is_is_active = True
                admin.is_superuser = is_admin
                admin.set_password(password)
                admin.save()
                print('Django superuser with username {0} successfully created'.format(username))
        except Exception, ex:
            sys.stderr.write('Error on superuser creation: {0}\n'.format(ex))
            sys.exit(1)

    def disable_zoo_in_folder(self, path, handler='django.project'):
        self.save_file(os.path.join(path, 'web.config'), self.DISABLE_ZOO_WEB_CONFIG.format(handler))

TRAC_ENV_FOLDER = 'trac'
TRAC_PROJECT_NAME = 'My Project'
TRAC_DB = 'sqlite:db/trac.db'

def main():
    d = Deployer()
    d.make_virtualenv()
    d.install_requirements()

    if os.path.exists(d.path_in_app(TRAC_ENV_FOLDER)):
        print("Trac env {0} already initialized.")
    else:
        d.system('cmd /c {0} {1} initenv "{2}" "{3}"'.format(
                    os.path.join(d.scripts_folder, 'trac-admin.exe'),
                    TRAC_ENV_FOLDER,
                    TRAC_PROJECT_NAME,
                    TRAC_DB
                ))
    print("Deploy sucessfully completed. Application is staring.")    

if __name__=='__main__':
    main()
