import sys
import os
import os.path
import shutil
import msvcrt
import time
import re
import urllib2
from contextlib import closing
import tarfile
import errno
from subprocess import call

class Deployer:

    class Unbuffered:
        def __init__(self, stream):
            self.stream = stream

        def write(self, data):
            self.stream.write(data)
            self.stream.flush()

        def __getattr__(self, attr):
            return getattr(self.stream, attr)

    DOWNLOAD_CHUNK = 1024*1024
    DOWNLOAD_PROGRESS_STEP_DOT = 1*1024*1024
    DOWNLOAD_PROGRESS_NUM_EVERY_DOT = 10

    DISABLE_ZOO_WEB_CONFIG = """<?xml version="1.0" encoding="UTF-8"?>
<configuration>
    <system.webServer>
        <handlers>
            <remove name="{0}#x86" />
            <remove name="{0}#x64" />
        </handlers>
    </system.webServer>
</configuration>
"""

    def __init__(self):
        try:
            self._disable_std_streams_buffering()
        except Exception:
            pass

        self.app_dir = os.environ.get('APPL_PHYSICAL_PATH') or os.path.dirname(__file__)
        self.virtualenv_exe = '"' + os.path.join(os.path.dirname(sys.executable), 'scripts\\virtualenv.exe') + '"'
        self.virtualenv_name = 'venv'
        self.virtualenv_dir = os.path.join(self.app_dir, self.virtualenv_name)
        self.local_python = '"' + os.path.join(self.virtualenv_dir, 'scripts', 'python.exe') + '"'
        self.pip_exe = '"' + os.path.join(self.virtualenv_dir, 'scripts', 'pip.exe') + '"'
        self.easy_install_exe = '"' + os.path.join(self.virtualenv_dir, 'scripts', 'easy_install.exe') + '"'

        self.last_progress_print = 0
        self.dirs = []

        os.chdir(self.app_dir)

        # update APPDATA env for pip
        os.environ['APPDATA'] = self.virtualenv_dir

        # add to python path:
        p1 = os.path.abspath(os.path.join(self.virtualenv_dir, 'Lib'))
        sys.path.insert(0, p1)
        os.environ['PYTHONPATH'] = p1 + ';' + os.environ.get('PYTHONPATH', '')
        p2 = os.path.abspath(os.path.join(self.virtualenv_dir, 'Lib', 'site-packages'))
        sys.path.insert(0, p2)
        os.environ['PYTHONPATH'] = p2 + ';' + os.environ.get('PYTHONPATH', '')

        # add scripts folder to PATH
        self.scripts_folder = os.path.abspath(os.path.join(self.virtualenv_dir, 'Scripts'))
        if os.environ.get('PATH'):
            os.environ['PATH'] = self.scripts_folder + ';' + os.environ['PATH']
        
        os.environ['VIRTUAL_ENV'] = self.virtualenv_dir

    def make_virtualenv(self, name=None):
        if not name:
            name = self.virtualenv_name
        if not os.path.exists(name):
            self.system(self.virtualenv_exe + ' ' + name)

    def _disable_std_streams_buffering(self):
        sys.stdout = Deployer.Unbuffered(sys.stdout)
        sys.stderr = Deployer.Unbuffered(sys.stderr)
        msvcrt.setmode(sys.stdout.fileno(), os.O_BINARY)
        msvcrt.setmode(sys.stderr.fileno(), os.O_BINARY)

    def mkdir_p(self, path):
        try:
            os.makedirs(path)
        except OSError as exc:
            if exc.errno == errno.EEXIST:
                pass
            else:
                raise

    def pushd(self, dirname):
        self.dirs.append(os.getcwd())
        os.chdir(dirname)

    def popd(self):
        if self.dirs:
            os.chdir(self.dirs.pop())

    def rmfile(self, file):
        if os.path.exists(file):
            os.remove(file)

    def system(self, command):
        print('\nRunning command: '+command)
        status = call(command, shell=True)
        if status != 0:
            sys.stderr.write('Command "{0}" returns exit code {1}'.format(command, status))
            sys.exit(status)

    def run_with_local_python(self, script):
        self.system(self.local_python + ' ' + script)

    def path_in_app(self, *args):
        return os.path.join(self.app_dir, *args)

    def install_requirements(self, requirements_path='requirements.txt', allow_external=False):
        print("Installing requirements")
        self.system('{0} install {1} --requirement="{2}"'.format(
            self.pip_exe,
            '--allow-external' if allow_external else '',
            requirements_path
        ))

    def easy_install(self, module_name):
        print("easy_install {0}".format(module_name))
        self.system('{0} {1}'.format(
            self.easy_install_exe,
            module_name
        ))

    def pip_install(self, module_name):
        print("pip install --allow-external {0}".format(module_name))
        self.system('{0} install {1}'.format(
            self.pip_exe,
            module_name
        ))

    def save_file(self, path, content):
        p = self.path_in_app(path)
        print("Updating file {0}".format(p))
        with open(p, 'wb') as f:
            f.write(content)

    def get_filename_from_url(self, url):
        if url.find('/') > 0:
            filename = url.split('/')[-1]
            if len(filename) > 4:
                return filename
        return re.sub(r'[-/:_\?&]+', '_', url)

    def download(self, url, filename=None):
        target = self.path_in_app(filename or self.get_filename_from_url(url))
        if os.path.exists(target):
            print('File {0} already exists. Download is skipped.'.format(target))
            return target
        print("Downloading "+url)
        chunk=None
        total = 0
        self.last_progress_print = 0
        with closing(urllib2.urlopen(url)) as furl:
            with open(target, 'wb') as fout:
                while True:
                    chunk = furl.read(self.DOWNLOAD_CHUNK)
                    if not chunk:
                        break
                    fout.write(chunk)
                    total += len(chunk)
                    self._print_download_progress(total)
        print('\nDownloaded {0:.1f} Mb'.format(total/1024.0/1024.0))
        print('Saved to '+target)
        return target

    def _print_download_progress(self, bytes_downloaded):
        delta_dots = (bytes_downloaded - self.last_progress_print*self.DOWNLOAD_PROGRESS_STEP_DOT)/self.DOWNLOAD_PROGRESS_STEP_DOT
        if delta_dots > 0:
            while delta_dots:
                delta_dots -= 1
                sys.stdout.write('.')
                self.last_progress_print += 1
                if self.last_progress_print % self.DOWNLOAD_PROGRESS_NUM_EVERY_DOT == 0:
                    sys.stdout.write(' {0} Mb\n'.format(bytes_downloaded/1024/1024))
                sys.stdout.flush()

    def create_django_user(self, settings, username, password, is_admin=True):
        try:
            from django.core.management import setup_environ
            setup_environ(settings)
            from django.contrib.auth.models import User
            # check superuser exists
            admins = User.objects.filter(username=username)
            if admins.count() >= 1:
                print('Django user already exists: {0}'.format(admins))
            else:
                admin = User()
                admin.username = username
                admin.is_staff = is_admin
                admin.is_is_active = True
                admin.is_superuser = is_admin
                admin.set_password(password)
                admin.save()
                print('Django superuser with username {0} successfully created'.format(username))
        except Exception, ex:
            sys.stderr.write('Error on superuser creation: {0}\n'.format(ex))
            sys.exit(1)

    def disable_zoo_in_folder(self, path, handler='django.project'):
        self.save_file(os.path.join(path, 'web.config'), self.DISABLE_ZOO_WEB_CONFIG.format(handler))

    def backup_file(self, filename):
        path = self. path_in_app(filename)
        backup_path = path+'~'
        shutil.copy(path, backup_path)
 
    def replace_in_file(self, filename, pattern, replace):
        path = self.path_in_app(filename)
        print('Fix file '+path)
        print('"{0}" -> "{1}"'.format(pattern, replace))
        with open(path, 'r') as f:
            s = f.read()
        (new_string, number_of_subs_made) = re.subn(pattern, replace, s, count=0, flags=re.M|re.I|re.U)
        s = new_string
        print('\t{0} subs was made'.format(number_of_subs_made))
        with open(path, 'w') as f:
            f.write(s)

    def hide_env_variable(self, name):
        new_name = 'HIDDEN_'+name
        os.environ[new_name] = os.environ[name]
        del os.environ[name]

    def show_env_variable(self, name):
        hidden_name = 'HIDDEN_'+name
        os.environ[name] = os.environ[hidden_name]
        del os.environ[hidden_name]


def main():
    """
    Deploy plan from http://djangobb.org/wiki/QuickInstall

    wget https://bitbucket.org/slav0nic/djangobb/get/stable.tar.gz
    tar zxvf stable.tar.gz
    virtualenv .env
    cd <place_for_virutalenv_dir>
    source .env/bin/activate
    # setup.py from djangobb app
    ./setup.py install
    # ./setup.py develop will be ok too if you are planning to upgrade djangobb from hg

    wget https://bitbucket.org/slav0nic/djangobb_project/get/tip.tar.gz
    tar zxvf tip.tar.gz
    cd slav0nic-djangobb_project-tip/
    pip install -r requirements.txt
    cd basic_project/
    touch local_settings.py
    # set DATABASE
    ./manage.py syncdb --all
    ./manage.py collectstatic
    ./manage.py runserver
    """

    SOURCE_FOLDER = 'slav0nic-djangobb-source'
    LOCAL_SETTINGS = """import os.path

DATABASES = {
    "default": {
        # Ends with "postgresql_psycopg2", "mysql", "sqlite3" or "oracle".
        "ENGINE": "django.db.backends.sqlite3",
        # DB name or path to database file if using sqlite3.
        "NAME": os.path.join(os.path.abspath(os.path.dirname(__file__)), "dev.db"),
        # Not used with sqlite3.
        "USER": "",
        # Not used with sqlite3.
        "PASSWORD": "",
        # Set to empty string for localhost. Not used with sqlite3.
        "HOST": "",
        # Set to empty string for default. Not used with sqlite3.
        "PORT": "",
    }
}

LOGGING = {
    'version': 1,
    'disable_existing_loggers': False,
    'handlers': {
        'mail_admins': {
            'level': 'ERROR',
            'class': 'django.utils.log.AdminEmailHandler'
        }
    },
    'loggers': {
        'django.request': {
            'handlers': ['mail_admins'],
            'level': 'ERROR',
            'propagate': True,
        },
    }
}
"""

    d = Deployer()

    d.make_virtualenv()

    #force install django < 1.6
    d.pip_install("django==1.5.5")
    d.pip_install("linaro-django-pagination")

    # wget https://bitbucket.org/slav0nic/djangobb/get/stable.tar.gz
    djbb_source = d.download('https://bitbucket.org/slav0nic/djangobb/get/stable.tar.gz', 'djbb_source.tar.gz')
    
    # tar zxvf stable.tar.gz
    print('Extracting '+djbb_source)
    with tarfile.open(djbb_source) as tar:
        for item in tar:
            # rename slav0nic-djangobb-70805f24fccc folder to slav0nic-djangobb-source
            item.name = SOURCE_FOLDER + item.name[item.name.find('/'):]
            tar.extract(item)

    d.pushd(SOURCE_FOLDER)

    source_requirements_file = os.path.join(SOURCE_FOLDER, 'extras', 'requirements.txt')

    # exclude PIL from requirements
    d.replace_in_file(source_requirements_file, r'^PIL', r'#PIL')
    d.replace_in_file(source_requirements_file, r'^-e\s+git', r'#-e git')

    # install pillow instead of PIL
    d.easy_install('pillow==2.1.0')
    d.easy_install('linaro_django_pagination==2.0.2')


    # now install requirements
    d.install_requirements('extras\\requirements.txt', allow_external=True)

    # run setup install without DJANGO_SETTINGS_MODULE env variable
    d.hide_env_variable('DJANGO_SETTINGS_MODULE')
    d.run_with_local_python('setup.py install')
    d.show_env_variable('DJANGO_SETTINGS_MODULE')

    d.popd()

    # clean up
    print('Removing '+SOURCE_FOLDER)
    shutil.rmtree(SOURCE_FOLDER)

    # wget https://bitbucket.org/slav0nic/djangobb_project/get/tip.tar.gz
    djbb_project = d.download('https://bitbucket.org/slav0nic/djangobb_project/get/tip.tar.gz', 'djbb_basic_project.tar.gz')
    
    # tar zxvf tip.tar.gz
    print('Extracting '+djbb_project)
    with tarfile.open(djbb_project, 'r:gz') as tar:
        for item in tar:
            # trim slav0nic-djangobb_project-548e4081d215 folder
            item.name = item.name[item.name.find('/')+1:]
            tar.extract(item)

    # pip install -r requirements.txt
    d.install_requirements()

    # cd basic_project/
    d.pushd('basic_project')

    # set DATABASE
    d.save_file(d.path_in_app('basic_project', 'local_settings.py'), LOCAL_SETTINGS)

    # collect static
    d.system('manage.py collectstatic -v 0 --noinput')

    # disable zoo in static folder
    d.disable_zoo_in_folder('basic_project\\static')

    # sync db & migrate
    d.system('manage.py syncdb --noinput --all')
    d.system('manage.py migrate --fake --noinput')

    d.popd()

    # call to create superuser
    d.run_with_local_python('"{0}" --create-django-user'.format(sys.argv[0]))

    # exit
    print("Deploy successfully completed. Application is staring.")


def create_django_user():
    d = Deployer()
    sys.path.insert(0, d.path_in_app('basic_project'))
    from basic_project import settings
    d.create_django_user(settings, 'admin', 'admin')

if __name__=='__main__':
    if len(sys.argv) == 2 and sys.argv[1] == '--create-django-user':
        create_django_user()
    else:
        main()
